import * as state from './state.js';
import * as ui from './ui.js';

import { fadeOutAndStop, backgroundMusic, t } from './utils.js';

let onProcessEndedCallback = () => {};


export const getInitialData = (platformId) => window.electronAPI.getInitialData(platformId);
export const getSettings = () => window.electronAPI.getSettings();
export const getLobbyList = () => window.electronAPI.getLobbyList();
export const manageSaveState = (options) => window.electronAPI.manageSaveState(options);
const getLocaleDataInternal = (lang) => window.electronAPI.getLocaleData(lang); 
export const getRelayConfig = () => window.electronAPI.getRelayConfig();
export const saveAppSettings = (settings) => window.electronAPI.saveAppSettings(settings);
export const saveAllSettings = (settings) => window.electronAPI.saveAllSettings(settings);
export const saveSettingsOnExit = (settings) => window.electronAPI.saveSettingsOnExit(settings);
export const setRelayConfig = (config) => window.electronAPI.setRelayConfig(config);
export const relaunchApp = () => window.electronAPI.relaunchApp();
export const launchGame = (options) => window.electronAPI.launchGame(options);
export const saveGame = (gamePath) => window.electronAPI.saveGame(gamePath);
export const downloadCollection = (platform) => window.electronAPI.downloadCollection(platform);
export const netplayHost = (options) => window.electronAPI.netplayHost(options);
export const joinNetplayGame = (options) => window.electronAPI.joinNetplayGame(options);
export const setFullscreen = (isFullscreen) => window.electronAPI.setFullscreen(isFullscreen);
export const exitApp = () => window.electronAPI.exitApp();
export const suspendPc = () => window.electronAPI.suspendPc();
export const powerOffPc = () => window.electronAPI.powerOffPc();
export const cancelEmuProcess = () => window.electronAPI.cancelEmuProcess();
export const openExternalLink = (url) => window.electronAPI.openExternalLink(url);

export async function loadTranslations(lang) {
    try {
        const translations = await getLocaleDataInternal(lang);
        state.setCurrentTranslations(translations);
        return translations; 
    } catch (e) {
        console.error(`Failed to load translations for ${lang}`, e);
        state.setCurrentTranslations({});
        return {}; 
    }
}

export function init(callbacks) {
    onProcessEndedCallback = callbacks.onProcessEnded;

    window.electronAPI.onEmuProgress((line) => {
        try {
            const data = JSON.parse(line);
            if (data.type === 'status') {
                let statusText;
                if (data.key) {
                    const replacements = { completed: data.completed, total: data.total, gameName: data.gameName };
                    statusText = t(data.key, replacements);
                } else {
                    statusText = data.message;
                }
                ui.progressStatus.textContent = statusText;
                if (data.key === 'progress_launching') {
                    fadeOutAndStop(backgroundMusic);
                }
            } else if (data.type === 'progress') {
                ui.progressBar.style.width = `${data.percent}%`;
                ui.progressPercent.textContent = `${data.percent}%`;
            } else if (data.type === 'error') {
                const errorText = data.key ? t(data.key, { message: data.message }) : t('progress_error', { message: data.message });
                ui.progressStatus.textContent = errorText;
                
                setTimeout(() => {
                    ui.hideProgressOverlay();
                    state.setProcessActive(false);
                    onProcessEndedCallback();
                }, 3000);
            }
        } catch (e) {
            console.log("Backend log:", line);
        }
    });

    window.electronAPI.onEmuExit((code) => {
        setTimeout(() => {
            ui.hideProgressOverlay();
            if (state.appSettings.bgmPath && state.appSettings.bgmPath !== "off") {
                backgroundMusic.currentTime = 0;
                backgroundMusic.play().catch(e => {});
            }
            
            const gameBaseName = state.games[state.selectedIndex].zipName.replace(/\.(zip|chd)$/i, '');
            delete state.saveStateCache[gameBaseName];
            ui.updateSaveStateIcon(state.selectedIndex);
            
            state.setProcessActive(false);
            onProcessEndedCallback();
        }, 500);
    });
}